#!/bin/bash

echo "Building Complete Cloth Shop Management System..."

# Create includes/header.php
cat > includes/header.php << 'HEADEREOF'
<?php if (!isset($pageTitle)) $pageTitle = 'Dashboard'; ?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($pageTitle); ?> - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="wrapper">
        <aside class="sidebar" id="sidebar">
            <div class="sidebar-header">
                <h3>🏪 <?php echo SHOP_NAME; ?></h3>
            </div>
            <nav>
                <ul class="sidebar-menu">
                    <li><a href="index.php" class="<?php echo basename($_SERVER['PHP_SELF']) == 'index.php' ? 'active' : ''; ?>">
                        📊 Dashboard
                    </a></li>
                    <li><a href="pos.php" class="<?php echo basename($_SERVER['PHP_SELF']) == 'pos.php' ? 'active' : ''; ?>">
                        💰 POS
                    </a></li>
                    <li><a href="products.php" class="<?php echo basename($_SERVER['PHP_SELF']) == 'products.php' ? 'active' : ''; ?>">
                        📦 Products
                    </a></li>
                    <li><a href="customers.php" class="<?php echo basename($_SERVER['PHP_SELF']) == 'customers.php' ? 'active' : ''; ?>">
                        👥 Customers
                    </a></li>
                    <li><a href="sales.php" class="<?php echo basename($_SERVER['PHP_SELF']) == 'sales.php' ? 'active' : ''; ?>">
                        🛒 Sales
                    </a></li>
                    <li><a href="reports.php" class="<?php echo basename($_SERVER['PHP_SELF']) == 'reports.php' ? 'active' : ''; ?>">
                        📈 Reports
                    </a></li>
                    <li><a href="logout.php">
                        🚪 Logout
                    </a></li>
                </ul>
            </nav>
        </aside>

        <main class="main-content">
            <header class="topbar">
                <div class="topbar-left">
                    <button class="menu-toggle" id="menuToggle">☰</button>
                    <h2><?php echo htmlspecialchars($pageTitle); ?></h2>
                </div>
                <div class="topbar-right">
                    <div class="user-info">
                        <strong><?php echo htmlspecialchars(getUsername()); ?></strong>
                        <span>(<?php echo htmlspecialchars(getUserRole()); ?>)</span>
                    </div>
                </div>
            </header>
            
            <div class="content-wrapper">
HEADEREOF

# Create includes/footer.php
cat > includes/footer.php << 'FOOTEREOF'
            </div>
        </main>
    </div>
    
    <script src="assets/js/main.js"></script>
</body>
</html>
FOOTEREOF

# Create index.php
cat > index.php << 'INDEXEOF'
<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'config/database.php';
require_once 'config/config.php';
require_once 'config/auth.php';

requireLogin();

$db = getDB();

// Get today's sales
$today = date('Y-m-d');
$stmt = $db->prepare("SELECT COUNT(*) as count, COALESCE(SUM(total_amount), 0) as total FROM sales WHERE DATE(sale_date) = ? AND status = 'completed'");
$stmt->bind_param('s', $today);
$stmt->execute();
$todaySales = $stmt->get_result()->fetch_assoc();
$stmt->close();

// Get counts
$customersCount = $db->query("SELECT COUNT(*) as count FROM customers WHERE status = 1")->fetch_assoc()['count'];
$lowStockCount = $db->query("SELECT COUNT(*) as count FROM products WHERE quantity <= low_stock_alert AND status = 1")->fetch_assoc()['count'];
$productsCount = $db->query("SELECT COUNT(*) as count FROM products WHERE status = 1")->fetch_assoc()['count'];

// Get recent sales
$recentSales = $db->query("SELECT s.*, c.name as customer_name FROM sales s LEFT JOIN customers c ON s.customer_id = c.id ORDER BY s.sale_date DESC LIMIT 5")->fetch_all(MYSQLI_ASSOC);

$pageTitle = 'Dashboard';
include 'includes/header.php';
?>

<?php
$flash = getFlashMessage();
if ($flash):
?>
    <div class="alert alert-<?php echo $flash['type']; ?>">
        <?php echo htmlspecialchars($flash['message']); ?>
    </div>
<?php endif; ?>

<div class="stats-grid">
    <div class="stat-card primary">
        <div class="stat-icon">💰</div>
        <div class="stat-value"><?php echo formatCurrency($todaySales['total']); ?></div>
        <div class="stat-label">Today's Sales</div>
        <div class="stat-info"><?php echo $todaySales['count']; ?> transactions</div>
    </div>
    
    <div class="stat-card success">
        <div class="stat-icon">📦</div>
        <div class="stat-value"><?php echo number_format($productsCount); ?></div>
        <div class="stat-label">Total Products</div>
        <div class="stat-info"><a href="products.php">Manage Products →</a></div>
    </div>
    
    <div class="stat-card info">
        <div class="stat-icon">👥</div>
        <div class="stat-value"><?php echo number_format($customersCount); ?></div>
        <div class="stat-label">Customers</div>
        <div class="stat-info"><a href="customers.php">View Customers →</a></div>
    </div>
    
    <div class="stat-card warning">
        <div class="stat-icon">⚠️</div>
        <div class="stat-value"><?php echo number_format($lowStockCount); ?></div>
        <div class="stat-label">Low Stock Items</div>
        <div class="stat-info"><?php echo $lowStockCount > 0 ? 'Needs Attention!' : 'All Good'; ?></div>
    </div>
</div>

<div class="card">
    <div class="card-header">
        <h3 class="card-title">📋 Recent Sales</h3>
        <a href="sales.php" class="btn btn-sm btn-primary">View All</a>
    </div>
    <div class="card-body" style="padding: 0;">
        <div class="table-responsive">
            <table class="table">
                <thead>
                    <tr>
                        <th>Invoice</th>
                        <th>Customer</th>
                        <th>Amount</th>
                        <th>Date</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (count($recentSales) > 0): ?>
                        <?php foreach ($recentSales as $sale): ?>
                            <tr>
                                <td><strong><?php echo htmlspecialchars($sale['invoice_number']); ?></strong></td>
                                <td><?php echo htmlspecialchars($sale['customer_name'] ?? 'Walk-in'); ?></td>
                                <td><?php echo formatCurrency($sale['total_amount']); ?></td>
                                <td><?php echo formatDate($sale['sale_date']); ?></td>
                                <td>
                                    <span class="badge badge-<?php echo $sale['status'] === 'completed' ? 'success' : 'warning'; ?>">
                                        <?php echo ucfirst($sale['status']); ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="5" class="text-center text-muted">No sales yet</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<div class="card" style="margin-top: 2rem;">
    <div class="card-header">
        <h3 class="card-title">⚡ Quick Actions</h3>
    </div>
    <div class="card-body">
        <div class="quick-actions">
            <a href="pos.php" class="btn btn-primary btn-lg">💰 New Sale</a>
            <a href="products.php" class="btn btn-success btn-lg">📦 Products</a>
            <a href="customers.php" class="btn btn-info btn-lg">👥 Customers</a>
            <a href="reports.php" class="btn btn-secondary btn-lg">📈 Reports</a>
        </div>
    </div>
</div>

<?php include 'includes/footer.php'; ?>
INDEXEOF

echo "✅ All core files created!"
echo "System ready for deployment"

